async function getCurrentTab() {
  const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
  return tab;
}

function getDomainFromUrl(url) {
  try {
    const urlObj = new URL(url);
    return urlObj.hostname;
  } catch {
    return 'unknown';
  }
}

async function loadPopupData() {
  const tab = await getCurrentTab();
  const currentDomain = getDomainFromUrl(tab.url);
  
  document.getElementById('site-url').textContent = currentDomain;

  chrome.tabs.sendMessage(tab.id, { action: 'getDetectedAI' }, (response) => {
    const detectedToolsDiv = document.getElementById('detected-tools');
    
    if (chrome.runtime.lastError || !response || !response.tools || response.tools.length === 0) {
      detectedToolsDiv.innerHTML = '<p class="no-detection">✓ No AI tools detected</p>';
    } else {
      detectedToolsDiv.innerHTML = response.tools.map(tool => 
        `<span class="tool-badge">${tool.name}</span>`
      ).join('');
    }
  });

  chrome.runtime.sendMessage({ action: 'getSettings' }, (settings) => {
    if (!settings) return;

    const allowedList = document.getElementById('allowed-list');
    const blockedList = document.getElementById('blocked-list');

    if (settings.allowedDomains.length === 0) {
      allowedList.innerHTML = '<p class="empty-message">No allowed sites yet</p>';
    } else {
      allowedList.innerHTML = settings.allowedDomains.map(domain => `
        <div class="domain-item">
          <span class="domain-name">${domain}</span>
          <button class="remove-btn" data-domain="${domain}" data-type="allowed">Remove</button>
        </div>
      `).join('');
    }

    if (settings.blockedDomains.length === 0) {
      blockedList.innerHTML = '<p class="empty-message">No blocked sites yet</p>';
    } else {
      blockedList.innerHTML = settings.blockedDomains.map(domain => `
        <div class="domain-item">
          <span class="domain-name">${domain}</span>
          <button class="remove-btn" data-domain="${domain}" data-type="blocked">Remove</button>
        </div>
      `).join('');
    }

    document.querySelectorAll('.remove-btn').forEach(btn => {
      btn.addEventListener('click', (e) => {
        const domain = e.target.getAttribute('data-domain');
        chrome.runtime.sendMessage({ 
          action: 'removeDomain', 
          domain 
        }, () => {
          loadPopupData();
        });
      });
    });

    const behaviorToggle = document.getElementById('behavior-detection-toggle');
    behaviorToggle.checked = settings.behaviorDetectionEnabled !== false;
    
    behaviorToggle.addEventListener('change', (e) => {
      chrome.runtime.sendMessage({
        action: 'setBehaviorDetection',
        enabled: e.target.checked
      });
    });
  });
}

document.addEventListener('DOMContentLoaded', loadPopupData);
