console.log('[Valerie] Content script loaded - v1.6.0');

const AI_WEBSITES = {
  chatgpt_site: {
    name: "ChatGPT",
    domains: ['chat.openai.com', 'chatgpt.com']
  },
  claude_site: {
    name: "Claude",
    domains: ['claude.ai']
  },
  gemini_site: {
    name: "Google Gemini",
    domains: ['gemini.google.com']
  },
  perplexity_site: {
    name: "Perplexity AI",
    domains: ['perplexity.ai']
  },
  poe_site: {
    name: "Poe",
    domains: ['poe.com']
  },
  character_ai_site: {
    name: "Character.AI",
    domains: ['character.ai', 'beta.character.ai']
  },
  bard_site: {
    name: "Google Bard",
    domains: ['bard.google.com']
  },
  copilot_site: {
    name: "Microsoft Copilot",
    domains: ['copilot.microsoft.com']
  },
  m365_copilot: {
    name: "Microsoft 365 Copilot",
    domains: ['m365.cloud.microsoft']
  },
  google_ai: {
    name: "Google AI Activity",
    domains: ['myactivity.google.com']
  }
};

const AI_SIGNATURES = {
  chatgpt: {
    name: "ChatGPT Widget",
    selectors: ['iframe[src*="chatgpt"]', 'div[class*="chatgpt"]', 'script[src*="openai"]'],
    scripts: ['openai.com', 'chat.openai.com'],
    confidence: 'confirmed-ai'
  },
  customgpt: {
    name: "Custom GPT Widget",
    selectors: ['iframe[src*="customgpt"]', 'div[class*="gpt-widget"]'],
    scripts: ['customgpt.ai'],
    confidence: 'confirmed-ai'
  },
  intercom: {
    name: "Intercom Chat",
    selectors: ['#intercom-container', 'iframe[name="intercom-messenger"]', '.intercom-app'],
    scripts: ['widget.intercom.io', 'js.intercomcdn.com'],
    confidence: 'suspect'
  },
  drift: {
    name: "Drift Chat",
    selectors: ['#drift-widget', 'iframe[name*="drift"]', '.drift-frame-controller'],
    scripts: ['js.driftt.com', 'customer.io'],
    confidence: 'suspect'
  },
  zendesk: {
    name: "Zendesk Chat",
    selectors: ['#launcher', 'iframe[id*="webWidget"]', '[data-test-id="launcher"]'],
    scripts: ['static.zdassets.com', 'zendesk.com'],
    confidence: 'suspect'
  },
  tidio: {
    name: "Tidio Chat",
    selectors: ['#tidio-chat', 'iframe[id*="tidio"]'],
    scripts: ['code.tidio.co'],
    confidence: 'suspect'
  },
  crisp: {
    name: "Crisp Chat",
    selectors: ['[data-crisp]', '.crisp-client'],
    scripts: ['client.crisp.chat'],
    confidence: 'suspect'
  },
  freshchat: {
    name: "Freshchat",
    selectors: ['#fc_frame', '.fc-widget'],
    scripts: ['wchat.freshchat.com'],
    confidence: 'suspect'
  },
  livechat: {
    name: "LiveChat",
    selectors: ['#chat-widget-container', 'iframe[id*="livechat"]'],
    scripts: ['cdn.livechatinc.com'],
    confidence: 'suspect'
  },
  helpscout: {
    name: "Help Scout Beacon",
    selectors: ['#beacon-container', 'iframe[name*="beacon"]'],
    scripts: ['beacon-v2.helpscout.net'],
    confidence: 'suspect'
  }
};

let detectedAI = [];
let suspectedChatbots = [];
let blockedDomains = [];
let allowedDomains = [];
let behaviorDetectionEnabled = true;

function getCurrentDomain() {
  return window.location.hostname;
}

async function loadSettings() {
  return new Promise((resolve) => {
    chrome.storage.local.get(['blockedDomains', 'allowedDomains', 'behaviorDetectionEnabled'], (result) => {
      blockedDomains = result.blockedDomains || [];
      allowedDomains = result.allowedDomains || [];
      behaviorDetectionEnabled = result.behaviorDetectionEnabled !== false;
      resolve();
    });
  });
}

function detectAIWebsite() {
  const currentDomain = getCurrentDomain();
  
  for (const [key, aiSite] of Object.entries(AI_WEBSITES)) {
    if (aiSite.domains.some(domain => currentDomain === domain || currentDomain.endsWith('.' + domain))) {
      return {
        id: key,
        name: aiSite.name,
        domain: currentDomain,
        isDirectSite: true
      };
    }
  }
  
  return null;
}

const AI_API_ENDPOINTS = [
  'api.openai.com',
  'api.anthropic.com',
  'generativelanguage.googleapis.com',
  'claude.ai/api',
  'chat.openai.com/backend-api',
  'bard.google.com/api',
  'api.perplexity.ai',
  'api.cohere.ai',
  'api.together.xyz',
  'api.replicate.com',
  'api.mistral.ai'
];

class BehavioralAIDetector {
  constructor() {
    this.aiScore = 0;
    this.indicators = [];
    this.networkActivityDetected = false;
    this.streamingDetected = false;
    this.websocketDetected = false;
    this.streamingNodes = new Map();
    this.widgetContainers = new Set();
    this.lastTextLength = new Map();
    this.textGrowthTimestamps = new Map();
    
    this.mouseActivity = {
      lastMovement: Date.now(),
      cooldownUntil: 0
    };
    
    this.typingActivity = {
      keyEvents: [],
      lastTyping: Date.now(),
      cooldownUntil: 0
    };
    
    this.scrollActivity = {
      lastScroll: Date.now(),
      cooldownUntil: 0,
      pageLoadTime: Date.now()
    };
  }

  monitorNetworkActivity() {
    const originalFetch = window.fetch;
    const originalXHR = window.XMLHttpRequest.prototype.open;
    const detector = this;

    window.fetch = async function(...args) {
      let url = args[0];
      
      if (url instanceof Request) {
        url = url.url;
      }
      
      const absoluteURL = detector.toAbsoluteURL(String(url));
      
      detector.checkURL(absoluteURL);
      
      return originalFetch.apply(this, args);
    };

    window.XMLHttpRequest.prototype.open = function(method, url) {
      const absoluteURL = detector.toAbsoluteURL(String(url));
      detector.checkURL(absoluteURL);
      return originalXHR.apply(this, arguments);
    };
  }

  toAbsoluteURL(url) {
    try {
      return new URL(url, window.location.href).href;
    } catch (e) {
      return url;
    }
  }

  checkURL(url) {
    if (AI_API_ENDPOINTS.some(endpoint => url.includes(endpoint))) {
      this.aiScore += 50;
      this.indicators.push('ai-api-call');
      this.networkActivityDetected = true;
      return;
    }
    
    if (url.includes('ogads-pa.clients6.google.com') && url.includes('AsyncData')) {
      this.aiScore += 50;
      this.indicators.push('google-ai-overview');
      this.networkActivityDetected = true;
      return;
    }
    
    if (url.includes('google.com')) {
      if (url.includes('/async/folsrch') || 
          url.includes('/async/bgasync') ||
          url.includes('rpcids=xAIAnswer') || 
          url.includes('ai_ans') || 
          url.includes('sge_creatives')) {
        this.aiScore += 50;
        this.indicators.push('google-ai-overview');
        this.networkActivityDetected = true;
      }
    }
  }

  monitorWebSockets() {
    const originalWebSocket = window.WebSocket;
    const detector = this;

    window.WebSocket = new Proxy(originalWebSocket, {
      construct(target, args) {
        const [url, protocols] = args;
        const absoluteURL = detector.toAbsoluteURL(String(url));
        
        if (AI_API_ENDPOINTS.some(endpoint => absoluteURL.includes(endpoint))) {
          detector.aiScore += 30;
          detector.indicators.push('websocket-connection');
          detector.websocketDetected = true;
        }
        
        return new target(url, protocols);
      },
      get(target, prop) {
        return Reflect.get(target, prop);
      }
    });
  }

  detectStreamingText() {
    const startObserving = () => {
      console.log('[Valerie] 👀 Starting streaming text observer');
      const observer = new MutationObserver((mutations) => {
        mutations.forEach((mutation) => {
          mutation.addedNodes.forEach((node) => {
            if (node.nodeType === Node.TEXT_NODE) {
              this.analyzeTextNode(node);
            } else if (node.nodeType === Node.ELEMENT_NODE) {
              const textContent = node.textContent || '';
              if (textContent.length > 100) {
                console.log(`[Valerie] Large DOM insertion detected: ${textContent.length} chars`);
                this.aiScore += 40;
                this.indicators.push('large-content-insertion');
                this.streamingDetected = true;
                this.streamingNodes.set(this.getNodeId(node), node);
                this.trackWidgetContainer(node);
              }
              
              const textNodes = this.getTextNodes(node);
              textNodes.forEach(textNode => this.analyzeTextNode(textNode));
            }
          });

          if (mutation.type === 'characterData') {
            this.analyzeTextNode(mutation.target);
          }
        });
      });

      observer.observe(document.body, {
        childList: true,
        subtree: true,
        characterData: true,
        characterDataOldValue: true
      });

      console.log('[Valerie] ✓ Streaming observer active');
      return observer;
    };

    if (document.body) {
      return startObserving();
    } else {
      console.log('[Valerie] Waiting for document.body...');
      const bodyWaiter = setInterval(() => {
        if (document.body) {
          clearInterval(bodyWaiter);
          startObserving();
        }
      }, 10);
    }
  }

  getTextNodes(element) {
    const textNodes = [];
    const walker = document.createTreeWalker(
      element,
      NodeFilter.SHOW_TEXT,
      null,
      false
    );

    let node;
    while (node = walker.nextNode()) {
      textNodes.push(node);
    }
    return textNodes;
  }

  analyzeTextNode(node) {
    if (!node.parentElement) return;

    const container = node.parentElement;
    const containerId = this.getNodeId(container);
    const currentText = node.textContent;
    const currentLength = currentText.length;

    if (!this.lastTextLength.has(containerId)) {
      this.lastTextLength.set(containerId, currentLength);
      this.textGrowthTimestamps.set(containerId, [Date.now()]);
      return;
    }

    const previousLength = this.lastTextLength.get(containerId);

    if (currentLength > previousLength) {
      const timestamps = this.textGrowthTimestamps.get(containerId) || [];
      timestamps.push(Date.now());

      if (timestamps.length >= 5) {
        const timeDeltas = [];
        for (let i = 1; i < timestamps.length; i++) {
          timeDeltas.push(timestamps[i] - timestamps[i-1]);
        }

        const avgDelta = timeDeltas.reduce((a, b) => a + b, 0) / timeDeltas.length;
        const totalGrowth = currentLength - (this.lastTextLength.get(containerId) - currentLength + previousLength);

        console.log(`[Valerie] Text growth: ${currentLength} chars, avgDelta: ${avgDelta}ms, growth: ${currentLength - previousLength}, score: ${this.aiScore}`);

        const isCharacterStreaming = avgDelta < 200 && avgDelta > 10 && currentLength > 50;
        const isChunkUpdate = currentLength > 100 && (currentLength - previousLength) > 20 && timestamps.length >= 3;
        
        if (isCharacterStreaming || isChunkUpdate) {
            this.aiScore += 40;
            this.indicators.push(isCharacterStreaming ? 'streaming-text-detected' : 'chunk-text-detected');
            this.streamingDetected = true;
            this.streamingNodes.set(containerId, container);
            this.trackWidgetContainer(container);
            console.log(`[Valerie] 🎯 AI DETECTED! Type: ${isCharacterStreaming ? 'Streaming' : 'Chunks'}, Score:`, this.aiScore);
          }
        }

      this.textGrowthTimestamps.set(containerId, timestamps.slice(-10));
    }

    this.lastTextLength.set(containerId, currentLength);
  }

  trackWidgetContainer(node) {
    if (!node) return;
    
    let current = node;
    let iterations = 0;
    const maxIterations = 50;
    
    while (current && iterations < maxIterations) {
      iterations++;
      
      if (current === document.body || current === document.documentElement) break;
      
      try {
        const style = window.getComputedStyle(current);
        const position = style.position;
        const zIndex = parseInt(style.zIndex) || 0;
        
        const isModal = current.getAttribute && (
          current.getAttribute('role') === 'dialog' || 
          current.getAttribute('aria-modal') === 'true'
        );
        const isHighZIndex = zIndex > 1000;
        const isFixedOrAbsolute = position === 'fixed' || position === 'absolute';
        const isIframe = current.tagName === 'IFRAME';
        const hasPopupSize = current.offsetWidth > 200 && current.offsetHeight > 200;
        
        if ((isFixedOrAbsolute && isHighZIndex && hasPopupSize) || isModal || isIframe) {
          this.widgetContainers.add(current);
          console.log('[Valerie] Tracked widget container:', current.tagName, current.className || current.id);
          
          if (current.parentElement && current.parentElement.tagName === 'IFRAME') {
            this.widgetContainers.add(current.parentElement);
          }
          break;
        }
      } catch (e) {
        console.log('[Valerie] Error checking element style:', e);
      }
      
      if (current.parentElement) {
        current = current.parentElement;
      } else {
        const root = current.getRootNode();
        if (root && root.host) {
          current = root.host;
        } else {
          break;
        }
      }
    }
  }

  getNodeId(node) {
    if (!node._valerie_id) {
      node._valerie_id = 'node_' + Math.random().toString(36).substr(2, 9);
    }
    return node._valerie_id;
  }

  monitorMouseActivity() {
    const handler = () => {
      this.mouseActivity.lastMovement = Date.now();
    };
    
    document.addEventListener('mousemove', handler, { passive: true });
    document.addEventListener('click', handler, { passive: true });
  }

  monitorTypingActivity() {
    const handler = (e) => {
      if (e.isTrusted) {
        const now = Date.now();
        this.typingActivity.keyEvents.push(now);
        this.typingActivity.lastTyping = now;
        
        if (this.typingActivity.keyEvents.length > 20) {
          this.typingActivity.keyEvents.shift();
        }
      }
    };
    
    document.addEventListener('keydown', handler, { passive: true });
  }

  monitorScrollActivity() {
    const handler = () => {
      this.scrollActivity.lastScroll = Date.now();
    };
    
    window.addEventListener('scroll', handler, { passive: true });
  }

  analyzeMouseBehavior() {
    const now = Date.now();
    
    if (now < this.mouseActivity.cooldownUntil) {
      return 0;
    }
    
    const timeSinceMovement = now - this.mouseActivity.lastMovement;
    const hasAIContent = this.streamingDetected || this.networkActivityDetected;
    
    if (hasAIContent && timeSinceMovement > 3000) {
      this.mouseActivity.cooldownUntil = now + 10000;
      this.indicators.push('no-mouse-ai-output');
      return 20;
    }
    
    return 0;
  }

  analyzeTypingBehavior() {
    const now = Date.now();
    
    if (now < this.typingActivity.cooldownUntil) {
      return 0;
    }
    
    if (this.typingActivity.keyEvents.length < 5) {
      return 0;
    }
    
    const recentEvents = this.typingActivity.keyEvents.filter(t => now - t < 2000);
    
    if (recentEvents.length >= 5) {
      const deltas = [];
      for (let i = 1; i < recentEvents.length; i++) {
        deltas.push(recentEvents[i] - recentEvents[i - 1]);
      }
      
      const avgDelta = deltas.reduce((a, b) => a + b, 0) / deltas.length;
      
      if (avgDelta < 100 && recentEvents.length >= 10) {
        this.typingActivity.cooldownUntil = now + 10000;
        this.indicators.push('instant-text-burst');
        return 25;
      }
    }
    
    return 0;
  }

  analyzeScrollBehavior() {
    const now = Date.now();
    
    if (now < this.scrollActivity.cooldownUntil) {
      return 0;
    }
    
    const timeSinceLoad = now - this.scrollActivity.pageLoadTime;
    const timeSinceScroll = now - this.scrollActivity.lastScroll;
    
    if (timeSinceLoad > 2000 && this.streamingDetected && timeSinceScroll > 3000) {
      this.scrollActivity.cooldownUntil = now + 10000;
      this.indicators.push('ai-without-scroll');
      return 15;
    }
    
    return 0;
  }

  getDetectionResult() {
    const threshold = 40;

    if (this.aiScore >= threshold) {
      return {
        id: 'behavior_detected',
        name: 'AI Activity (detected by behavior)',
        domain: getCurrentDomain(),
        isDirectSite: false,
        isBehaviorDetected: true,
        score: this.aiScore,
        indicators: this.indicators,
        details: {
          networkActivity: this.networkActivityDetected,
          streaming: this.streamingDetected,
          websocket: this.websocketDetected,
          mousePattern: this.indicators.includes('no-mouse-ai-output'),
          typingPattern: this.indicators.includes('instant-text-burst'),
          scrollPattern: this.indicators.includes('ai-without-scroll')
        }
      };
    }

    return null;
  }
}

let behavioralDetector = null;

function initBehavioralDetection() {
  behavioralDetector = new BehavioralAIDetector();
  behavioralDetector.monitorNetworkActivity();
  behavioralDetector.monitorWebSockets();
  behavioralDetector.detectStreamingText();
  behavioralDetector.monitorMouseActivity();
  behavioralDetector.monitorTypingActivity();
  behavioralDetector.monitorScrollActivity();
}

function checkBehavioralDetection() {
  if (behavioralDetector) {
    const mouseScore = behavioralDetector.analyzeMouseBehavior();
    const typingScore = behavioralDetector.analyzeTypingBehavior();
    const scrollScore = behavioralDetector.analyzeScrollBehavior();
    
    behavioralDetector.aiScore += mouseScore + typingScore + scrollScore;
    
    return behavioralDetector.getDetectionResult();
  }
  return null;
}

function showBlockedSiteOverlay(siteName) {
  const overlay = document.createElement('div');
  overlay.id = 'valerie-block-overlay';
  overlay.style.cssText = `
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background: #1a1a1a;
    z-index: 2147483647;
    display: flex;
    align-items: center;
    justify-content: center;
    font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
  `;

  const content = document.createElement('div');
  content.style.cssText = `
    text-align: center;
    color: white;
    max-width: 600px;
    padding: 40px;
  `;

  content.innerHTML = `
    <div style="font-size: 80px; margin-bottom: 20px;">🛡️</div>
    <h1 style="font-size: 32px; margin-bottom: 15px; font-weight: 600;">Site Blocked by Valerie</h1>
    <p style="font-size: 18px; color: #ccc; margin-bottom: 30px;">${siteName} has been blocked by your settings.</p>
    <button id="valerie-allow-site-btn" style="
      background: #4CAF50;
      color: white;
      border: none;
      padding: 15px 30px;
      font-size: 16px;
      border-radius: 8px;
      cursor: pointer;
      font-weight: 500;
      margin-right: 10px;
    ">Allow This Site</button>
    <button id="valerie-go-back-btn" style="
      background: #666;
      color: white;
      border: none;
      padding: 15px 30px;
      font-size: 16px;
      border-radius: 8px;
      cursor: pointer;
      font-weight: 500;
    ">Go Back</button>
  `;

  overlay.appendChild(content);
  document.documentElement.appendChild(overlay);

  document.getElementById('valerie-allow-site-btn').onclick = () => {
    chrome.runtime.sendMessage({
      action: 'allowDomain',
      domain: getCurrentDomain()
    }, () => {
      overlay.remove();
      location.reload();
    });
  };

  document.getElementById('valerie-go-back-btn').onclick = () => {
    history.back();
  };

  document.body.style.overflow = 'hidden';
}

function detectAITools() {
  const detected = [];
  const suspected = [];
  const currentDomain = getCurrentDomain();

  for (const [key, ai] of Object.entries(AI_SIGNATURES)) {
    let found = false;

    for (const selector of ai.selectors) {
      if (document.querySelector(selector)) {
        found = true;
        break;
      }
    }

    if (!found) {
      const scripts = document.querySelectorAll('script[src]');
      for (const script of scripts) {
        const src = script.getAttribute('src');
        if (src && ai.scripts.some(pattern => src.includes(pattern))) {
          found = true;
          break;
        }
      }
    }

    if (found) {
      const detection = {
        id: key,
        name: ai.name,
        domain: currentDomain,
        confidence: ai.confidence
      };
      
      if (ai.confidence === 'confirmed-ai') {
        detected.push(detection);
      } else if (ai.confidence === 'suspect') {
        suspected.push(detection);
      }
    }
  }

  return { confirmed: detected, suspected };
}

function removeAIWidget(aiId) {
  if (aiId === 'behavior_detected') {
    if (behavioralDetector) {
      let blockedCount = 0;
      
      if (behavioralDetector.widgetContainers) {
        const staleContainers = [];
        behavioralDetector.widgetContainers.forEach(container => {
          if (!container.isConnected) {
            staleContainers.push(container);
            return;
          }
          
          try {
            const wasVisible = container.style.display !== 'none';
            container.style.setProperty('display', 'none', 'important');
            container.style.setProperty('visibility', 'hidden', 'important');
            container.style.setProperty('opacity', '0', 'important');
            if (container.setAttribute) {
              container.setAttribute('aria-hidden', 'true');
            }
            if (wasVisible) {
              blockedCount++;
              console.log('[Valerie] Blocked widget container:', container.tagName, container.className || container.id);
            }
          } catch (e) {
            console.log('[Valerie] Error blocking container:', e);
          }
        });
        
        staleContainers.forEach(c => behavioralDetector.widgetContainers.delete(c));
      }
      
      if (behavioralDetector.streamingNodes) {
        behavioralDetector.streamingNodes.forEach((node, id) => {
          if (node && node.parentElement) {
            node.style.setProperty('display', 'none', 'important');
            node.style.setProperty('visibility', 'hidden', 'important');
            node.style.setProperty('opacity', '0', 'important');
          }
        });
      }
      
      if (blockedCount > 0) {
        console.log(`[Valerie] Blocked ${blockedCount} AI widget(s)`);
      }
    }
    return;
  }

  const ai = AI_SIGNATURES[aiId];
  if (!ai) return;

  ai.selectors.forEach(selector => {
    const elements = document.querySelectorAll(selector);
    elements.forEach(el => {
      el.style.display = 'none';
      el.remove();
    });
  });

  const scripts = document.querySelectorAll('script[src]');
  scripts.forEach(script => {
    const src = script.getAttribute('src');
    if (src && ai.scripts.some(pattern => src.includes(pattern))) {
      script.remove();
    }
  });
}

function showPrompt(aiTools) {
  if (document.getElementById('valerie-prompt-overlay')) {
    console.log('[Valerie] Prompt already showing, skipping duplicate');
    return;
  }
  
  const overlay = document.createElement('div');
  overlay.id = 'valerie-prompt-overlay';
  overlay.style.cssText = `
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background: rgba(0, 0, 0, 0.7);
    z-index: 999999;
    display: flex;
    align-items: center;
    justify-content: center;
    font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
  `;

  const prompt = document.createElement('div');
  prompt.style.cssText = `
    background: white;
    padding: 30px;
    border-radius: 12px;
    max-width: 500px;
    box-shadow: 0 10px 40px rgba(0, 0, 0, 0.3);
  `;

  const title = document.createElement('h2');
  title.textContent = '🛡️ Valerie AI Detector';
  title.style.cssText = 'margin: 0 0 15px 0; color: #333; font-size: 24px;';

  const message = document.createElement('p');
  const isDirectSite = aiTools.length === 1 && aiTools[0].isDirectSite;
  message.textContent = isDirectSite 
    ? `You are visiting an AI website:` 
    : `Detected ${aiTools.length} AI tool${aiTools.length > 1 ? 's' : ''} on this website:`;
  message.style.cssText = 'color: #666; margin-bottom: 15px;';

  const list = document.createElement('ul');
  list.style.cssText = 'margin: 10px 0 20px 20px; color: #333;';
  aiTools.forEach(tool => {
    const item = document.createElement('li');
    item.textContent = tool.name;
    item.style.cssText = 'margin: 5px 0;';
    list.appendChild(item);
  });

  const question = document.createElement('p');
  question.textContent = 'What would you like to do?';
  question.style.cssText = 'color: #333; font-weight: 500; margin-bottom: 20px;';

  const buttonContainer = document.createElement('div');
  buttonContainer.style.cssText = 'display: flex; gap: 10px;';

  const allowBtn = document.createElement('button');
  allowBtn.textContent = 'Allow';
  allowBtn.style.cssText = `
    flex: 1;
    padding: 12px;
    background: #4CAF50;
    color: white;
    border: none;
    border-radius: 6px;
    font-size: 16px;
    cursor: pointer;
    font-weight: 500;
  `;
  allowBtn.onmouseover = () => allowBtn.style.background = '#45a049';
  allowBtn.onmouseout = () => allowBtn.style.background = '#4CAF50';

  const blockBtn = document.createElement('button');
  blockBtn.textContent = 'Block';
  blockBtn.style.cssText = `
    flex: 1;
    padding: 12px;
    background: #f44336;
    color: white;
    border: none;
    border-radius: 6px;
    font-size: 16px;
    cursor: pointer;
    font-weight: 500;
  `;
  blockBtn.onmouseover = () => blockBtn.style.background = '#da190b';
  blockBtn.onmouseout = () => blockBtn.style.background = '#f44336';

  allowBtn.onclick = () => {
    chrome.runtime.sendMessage({
      action: 'allowDomain',
      domain: getCurrentDomain()
    });
    overlay.remove();
  };

  blockBtn.onclick = () => {
    const isDirectSite = aiTools.length === 1 && aiTools[0].isDirectSite;
    chrome.runtime.sendMessage({
      action: 'blockDomain',
      domain: getCurrentDomain()
    }, () => {
      if (isDirectSite) {
        overlay.remove();
        location.reload();
      } else {
        aiTools.forEach(tool => removeAIWidget(tool.id));
        overlay.remove();
      }
    });
  };

  buttonContainer.appendChild(allowBtn);
  buttonContainer.appendChild(blockBtn);

  prompt.appendChild(title);
  prompt.appendChild(message);
  prompt.appendChild(list);
  prompt.appendChild(question);
  prompt.appendChild(buttonContainer);
  overlay.appendChild(prompt);
  document.body.appendChild(overlay);
}

async function init() {
  console.log('[Valerie] init() called');
  await loadSettings();
  console.log('[Valerie] Settings loaded, behavior detection enabled:', behaviorDetectionEnabled);
  
  const currentDomain = getCurrentDomain();
  console.log('[Valerie] Current domain:', currentDomain);
  
  const aiWebsite = detectAIWebsite();
  if (aiWebsite) {
    if (blockedDomains.includes(currentDomain)) {
      showBlockedSiteOverlay(aiWebsite.name);
      return;
    }
    
    if (!allowedDomains.includes(currentDomain)) {
      detectedAI = [aiWebsite];
      showPrompt([aiWebsite]);
      chrome.runtime.sendMessage({
        action: 'aiDetected',
        tools: [aiWebsite],
        domain: currentDomain
      });
      return;
    }
  }
  
  if (allowedDomains.includes(currentDomain)) {
    return;
  }

  if (behaviorDetectionEnabled) {
    initBehavioralDetection();
  }

  setTimeout(() => {
    const detection = detectAITools();
    detectedAI = detection.confirmed;
    suspectedChatbots = detection.suspected;
    
    if (detectedAI.length > 0) {
      if (blockedDomains.includes(currentDomain)) {
        detectedAI.forEach(tool => removeAIWidget(tool.id));
      } else {
        showPrompt(detectedAI);
      }
      
      chrome.runtime.sendMessage({
        action: 'aiDetected',
        tools: detectedAI,
        domain: currentDomain
      });
    }
  }, 1500);

  setInterval(() => {
    if (blockedDomains.includes(currentDomain)) {
      removeAIWidget('behavior_detected');
      detectedAI.forEach(tool => {
        if (tool.id && !tool.isBehaviorDetected) {
          removeAIWidget(tool.id);
        }
      });
    }
    
    if (behaviorDetectionEnabled && !allowedDomains.includes(currentDomain)) {
      const behaviorDetection = checkBehavioralDetection();
      console.log('[Valerie] Behavior check - Result:', behaviorDetection ? `DETECTED (score: ${behaviorDetection.score})` : 'None');
      
      if (behaviorDetection) {
        const upgradedSuspects = [];
        
        if (suspectedChatbots.length > 0) {
          console.log('[Valerie] Upgrading suspected chatbots to confirmed AI based on behavioral detection');
          suspectedChatbots.forEach(suspect => {
            if (!detectedAI.some(tool => tool.id === suspect.id)) {
              const upgraded = {...suspect, confidence: 'confirmed-ai', upgradedByBehavior: true};
              detectedAI.push(upgraded);
              upgradedSuspects.push(upgraded);
            }
          });
          suspectedChatbots = [];
        }
        
        if (blockedDomains.includes(currentDomain)) {
          console.log('[Valerie] AI behavior detected on blocked domain - hiding content');
          removeAIWidget('behavior_detected');
          detectedAI.forEach(tool => removeAIWidget(tool.id));
        } else {
          const alreadyHasBehaviorDetection = detectedAI.some(tool => tool.isBehaviorDetected);
          
          if (!alreadyHasBehaviorDetection) {
            detectedAI.push(behaviorDetection);
          }
          
          if (upgradedSuspects.length > 0 || !alreadyHasBehaviorDetection) {
            console.log('[Valerie] Showing prompt for behavioral detection');
            
            const toolsToShow = alreadyHasBehaviorDetection
              ? upgradedSuspects
              : upgradedSuspects.concat([behaviorDetection]);
            
            if (toolsToShow.length > 0) {
              showPrompt(toolsToShow);
              chrome.runtime.sendMessage({
                action: 'aiDetected',
                tools: detectedAI,
                domain: currentDomain
              });
            }
          }
        }
      }
    }
  }, 1000);

  const observer = new MutationObserver(() => {
    const detection = detectAITools();
    const newConfirmed = detection.confirmed;
    suspectedChatbots = detection.suspected;
    
    if (blockedDomains.includes(currentDomain) && newConfirmed.length > 0) {
      newConfirmed.forEach(tool => removeAIWidget(tool.id));
    }
    
    const confirmedOnly = detectedAI.filter(t => !t.isBehaviorDetected && !t.upgradedByBehavior);
    
    if (newConfirmed.length > confirmedOnly.length) {
      const behaviorDetected = detectedAI.filter(t => t.isBehaviorDetected);
      const upgraded = detectedAI.filter(t => t.upgradedByBehavior);
      
      detectedAI = newConfirmed.concat(behaviorDetected).concat(upgraded);
    }
  });

  observer.observe(document.body, {
    childList: true,
    subtree: true
  });
}

console.log('[Valerie] Initializing... readyState:', document.readyState);

if (document.readyState === 'loading') {
  console.log('[Valerie] Waiting for DOMContentLoaded');
  document.addEventListener('DOMContentLoaded', () => {
    console.log('[Valerie] DOMContentLoaded fired, calling init()');
    init();
  });
} else {
  console.log('[Valerie] Document ready, calling init() immediately');
  init();
}

chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  if (message.action === 'getDetectedAI') {
    sendResponse({ tools: detectedAI });
  } else if (message.action === 'removeAI') {
    detectedAI.forEach(tool => removeAIWidget(tool.id));
  }
});
